/*
 * Decompiled with CFR 0.152.
 */
package com.semmle.util.files;

import com.semmle.util.basic.ObjectUtil;
import com.semmle.util.data.StringUtil;
import com.semmle.util.exception.CatastrophicError;
import com.semmle.util.exception.Exceptions;
import com.semmle.util.exception.ResourceError;
import com.semmle.util.io.csv.CSVReader;
import com.semmle.util.process.Env;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.lang.reflect.UndeclaredThrowableException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtil {
    private static final Logger logger;
    private static final Pattern rpInvalidFilenameCharacters;
    public static final Charset UTF8;
    private static final BitSet allowedCharacters;
    public static final FileFilter falseFilter;
    private static final int FILE_OPERATION_ATTEMPTS_LIMIT = 30;
    private static final int FILE_OPERATION_ATTEMPTS_DELAY_MS = 1000;

    public static File[] list(File f) {
        return FileUtil.list(f, null);
    }

    public static File[] list(File f, FileFilter filter) {
        Object[] files;
        Object[] objectArray = files = filter == null ? f.listFiles() : f.listFiles(filter);
        if (files == null) {
            boolean exists = f.exists();
            boolean isDirectory = f.isDirectory();
            boolean canRead = f.canRead();
            throw new ResourceError("Could not list the contents of directory " + f + " - " + (!exists ? "file does not exist." : (!isDirectory ? "file is not a directory." : (!canRead ? "cannot read - permission denied." : "unknown I/O error."))));
        }
        Arrays.sort(files);
        return files;
    }

    @Deprecated
    public static Set<File> recursiveFind(File dir, FileFilter filter) {
        LinkedHashSet<File> result = new LinkedHashSet<File>();
        FileUtil.recursiveFind(dir, filter, result);
        return result;
    }

    @Deprecated
    public static Set<File> recursiveFind(File dir, FileFilter filter, FileFilter recurseFilter) {
        LinkedHashSet<File> result = new LinkedHashSet<File>();
        FileUtil.recursiveFind(dir, filter, recurseFilter, result);
        return result;
    }

    @Deprecated
    public static void recursiveFind(File dir, FileFilter filter, Set<File> result) {
        FileUtil.recursiveFind(dir, filter, null, result);
    }

    @Deprecated
    public static void recursiveFind(File dir, FileFilter filter, FileFilter recurseFilter, Set<File> result) {
        for (File f : FileUtil.list(dir, filter)) {
            result.add(f);
        }
        FileFilter recurseDirFilter = recurseFilter == null ? FileUtil.kindFilter(false) : FileUtil.andFilters(FileUtil.kindFilter(false), recurseFilter);
        for (File f : FileUtil.list(dir, recurseDirFilter)) {
            FileUtil.recursiveFind(f, filter, recurseFilter, result);
        }
    }

    public static void mkdirs(File dir) {
        try {
            Files.createDirectories(dir.toPath(), new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException e) {
            throw new ResourceError("Can't create " + dir + " -- already exists as a file", e);
        }
        catch (IOException e) {
            throw new ResourceError("Can't create " + dir, e);
        }
    }

    public static boolean isAbsolute(String path) {
        if (path == null || path.length() == 0) {
            return false;
        }
        return path.charAt(0) == '/' || path.charAt(0) == '\\' || path.length() >= 3 && Character.isLetter(path.charAt(0)) && path.charAt(1) == ':' && (path.charAt(2) == '/' || path.charAt(2) == '\\');
    }

    public static boolean copy(File from, File to) {
        boolean targetFileExisted = to.exists();
        try {
            Files.copy(from.toPath(), to.toPath(), StandardCopyOption.REPLACE_EXISTING);
            return true;
        }
        catch (IOException e) {
            if (!targetFileExisted) {
                to.delete();
            }
            logger.error("Cannot copy " + from + " to " + to, e);
            return false;
        }
    }

    public static boolean append(File sourceFile, File targetFile) {
        try {
            sourceFile = FileUtil.tryMakeCanonical(sourceFile);
            targetFile = FileUtil.tryMakeCanonical(targetFile);
            if (sourceFile.equals(targetFile)) {
                logger.error("Trying to append the contents of file " + sourceFile + " onto itself");
                return false;
            }
            try (InputStream fis = Files.newInputStream(sourceFile.toPath(), new OpenOption[0]);
                 BufferedInputStream in = new BufferedInputStream(fis);
                 OutputStream fos = Files.newOutputStream(targetFile.toPath(), StandardOpenOption.APPEND, StandardOpenOption.CREATE);
                 BufferedOutputStream out = new BufferedOutputStream(fos);){
                int count;
                byte[] buf = new byte[16384];
                while ((count = ((InputStream)in).read(buf)) > 0) {
                    ((OutputStream)out).write(buf, 0, count);
                }
            }
            return true;
        }
        catch (IOException e) {
            logger.error("Cannot append contents of " + sourceFile + " to " + targetFile, e);
            return false;
        }
    }

    public static void write(File file, String content) {
        try (Writer writer = FileUtil.openWriterUTF8(file, true, false);){
            writer.write(content);
        }
        catch (IOException e) {
            throw new ResourceError("Failed to write to file " + file, e);
        }
    }

    public static void append(File file, String content) {
        try (Writer writer = FileUtil.openWriterUTF8(file, false, true);){
            writer.write(content);
        }
        catch (IOException e) {
            throw new ResourceError("Failed to append to file " + file, e);
        }
    }

    public static String readText(File f) throws IOException {
        return FileUtil.readText(f.toPath());
    }

    public static String readText(Path f) throws IOException {
        char[] cbuf = new char[10240];
        StringBuilder buf = new StringBuilder();
        try (BufferedReader reader = Files.newBufferedReader(f, UTF8);){
            int n = reader.read(cbuf);
            while (n > 0) {
                buf.append(cbuf, 0, n);
                n = reader.read(cbuf);
            }
        }
        return buf.toString();
    }

    public static final String sanitizeFilename(String name) {
        StringBuffer safe = new StringBuffer();
        for (int i = 0; i < name.length(); ++i) {
            char c = name.charAt(i);
            if (!allowedCharacters.get(c)) continue;
            safe.append(c);
        }
        return safe.toString();
    }

    public static final File createUniqueFile(File baseDirectory, String fileName) {
        return FileUtil.createUniqueFileImpl(baseDirectory, fileName, false);
    }

    public static final File createUniqueDirectory(File baseDirectory, String dirName) {
        return FileUtil.createUniqueFileImpl(baseDirectory, dirName, true);
    }

    private static final File createUniqueFileImpl(File baseDirectory, String fileName, boolean directory) {
        String baseName;
        if (!baseDirectory.exists()) {
            throw new IllegalArgumentException("FileUtil.makeUniqueName(" + baseDirectory + ",\"" + fileName + "\"):  directory " + baseDirectory + " does not exist.");
        }
        if (!baseDirectory.isDirectory()) {
            throw new IllegalArgumentException("FileUtil.makeUniqueName(" + baseDirectory + ",\"" + fileName + "\"):  file " + baseDirectory + " is not a directory.");
        }
        if (fileName.contains("/")) {
            throw new IllegalArgumentException("FileUtil.makeUniqueName(" + baseDirectory + ",\"" + fileName + "\"):  file name \"" + fileName + "\" is not a simple file name.");
        }
        String name = baseName = (fileName = FileUtil.replaceInvalidFilenameChars(fileName));
        File candidateFile = new File(baseDirectory, name);
        String extension = FileUtil.extension(new File(baseName));
        int i = 1;
        try {
            while (!(!directory ? candidateFile.createNewFile() : candidateFile.mkdir())) {
                name = extension.length() > 0 ? baseName.substring(0, baseName.length() - extension.length()) + "-" + i + extension : baseName + "-" + i;
                candidateFile = new File(baseDirectory, name);
                ++i;
            }
        }
        catch (IOException e) {
            throw new ResourceError("Failed to create a unique file in " + baseDirectory, e);
        }
        return candidateFile;
    }

    public static String replaceInvalidFilenameChars(String fileName) {
        return rpInvalidFilenameCharacters.matcher(fileName).replaceAll("_");
    }

    public static String extension(File f) {
        return FileUtil.extension(f.getName());
    }

    public static String extension(Path f) {
        return FileUtil.extension(f.getFileName().toString());
    }

    public static String extension(String name) {
        int i = name.lastIndexOf(46);
        if (i == -1) {
            return "";
        }
        String extension = name.substring(i);
        if (extension.equals(".gz") || extension.equals(".br")) {
            int before = name.lastIndexOf(46, i - 1);
            if (before == -1) {
                return extension;
            }
            String combinedExtension = name.substring(before);
            return combinedExtension;
        }
        if (extension.equals(".zip") && name.endsWith(".xml.zip")) {
            return ".xml.zip";
        }
        return extension;
    }

    public static String basename(File f) {
        return FileUtil.stripExtension(f.getName());
    }

    public static String basename(Path path) {
        Path filename = path.getFileName();
        return filename == null ? "" : FileUtil.stripExtension(filename.toString());
    }

    public static String stripExtension(String name) {
        return name.substring(0, name.length() - FileUtil.extension(name).length());
    }

    public static File withExtension(File f, String extension) {
        return new File(FileUtil.withExtension(f.getPath(), extension));
    }

    public static String withExtension(String path, String extension) {
        String oldExtension = FileUtil.extension(path);
        return path.substring(0, path.length() - oldExtension.length()) + extension;
    }

    public static FileFilter extensionFilter(boolean caseSensitive, String ... extensions) {
        return new FileFilterImpl(extensions, caseSensitive, true);
    }

    public static FileFilter kindFilter(final boolean files) {
        return new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                if (pathname.isFile() && files) {
                    return true;
                }
                return pathname.isDirectory() && !files;
            }
        };
    }

    public static FileFilter andFilters(final FileFilter ... filters) {
        return new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                for (FileFilter filter : filters) {
                    if (filter.accept(pathname)) continue;
                    return false;
                }
                return true;
            }
        };
    }

    private static String santizePathString(String pathString) {
        pathString = pathString.replace(':', '_');
        pathString = pathString.replace('\\', File.separatorChar).replace('/', File.separatorChar);
        return pathString;
    }

    public static File appendAbsolutePath(File root, String absolutePath) {
        absolutePath = FileUtil.santizePathString(absolutePath);
        return new File(root, absolutePath).getAbsoluteFile();
    }

    public static void close(Closeable resourceToClose) {
        if (resourceToClose != null) {
            try {
                resourceToClose.close();
            }
            catch (IOException ignored) {
                Exceptions.ignore(ignored, "Contract is to ignore");
            }
            catch (UndeclaredThrowableException maybeIgnored) {
                if (maybeIgnored.getCause() instanceof IOException) {
                    Exceptions.ignore(maybeIgnored, "Undeclared exception was an IOException, ignoring");
                }
                throw maybeIgnored;
            }
        }
    }

    public static ResolvedCompressedSourceArchivePaths resolveCompressedSourceArchivePaths(File srcArchiveZip, String convertedSourceLocationPrefix) {
        String sourcePath;
        String srcArchivePath = "";
        boolean legacyZip = srcArchiveZip.getName().equals("src_archive.zip");
        if (legacyZip) {
            srcArchivePath = FileUtil.convertPathForSourceArchiveZip("");
            sourcePath = srcArchivePath + convertedSourceLocationPrefix;
        } else {
            sourcePath = convertedSourceLocationPrefix.startsWith("/") ? convertedSourceLocationPrefix.substring(1) : convertedSourceLocationPrefix;
        }
        return new ResolvedCompressedSourceArchivePaths(srcArchivePath, sourcePath);
    }

    public static String convertAbsolutePathForSourceArchive(String absolutePath) {
        if (absolutePath == null) {
            throw new CatastrophicError("FileUtil.convertPathForSourceArchiveZip: absolutePath must be non-null");
        }
        absolutePath = FileUtil.normalisePath(absolutePath);
        if (!(absolutePath = absolutePath.replace(':', '_')).startsWith("/")) {
            absolutePath = "/" + absolutePath;
        }
        if (absolutePath.endsWith("/")) {
            absolutePath = absolutePath.substring(0, absolutePath.length() - 1);
        }
        return absolutePath;
    }

    public static String convertPathForSourceArchiveZip(String absolutePath) {
        return "src_archive" + FileUtil.convertAbsolutePathForSourceArchive(absolutePath);
    }

    public static File fileRelativeTo(File base, String relativePath) {
        if (!FileUtil.isRelativePath(relativePath)) {
            throw new CatastrophicError("Invalid relative path '" + relativePath + "'.");
        }
        return new File(base, relativePath);
    }

    public static boolean isRelativePath(String path) {
        if (path.startsWith("/")) {
            return false;
        }
        if (path.startsWith("\\")) {
            return false;
        }
        return Env.getOS() != Env.OS.WINDOWS || !path.contains(":");
    }

    public static String normalisePath(String path) {
        char driveLetter;
        if (path == null) {
            throw new CatastrophicError("FileUtil.normalisePath: path must be non-null");
        }
        if (path.length() >= 2 && path.charAt(1) == ':' && (driveLetter = path.charAt(0)) >= 'a' && driveLetter <= 'z') {
            path = Character.toUpperCase(driveLetter) + path.substring(1);
        }
        path = path.replace('\\', '/');
        return path;
    }

    public static String relativePath(File file, File baseDirectory) {
        String candidate = FileUtil.relativePathOpt(file, baseDirectory);
        if (candidate != null) {
            return candidate;
        }
        throw new ResourceError("Could not determine a relative path to " + file + " from " + baseDirectory);
    }

    public static String relativePathOpt(File file, File baseDirectory) {
        try {
            File canonicalToFile = file.getCanonicalFile();
            File canonicalToBase = baseDirectory.getCanonicalFile();
            String canonicalRelative = FileUtil.relativePathAsIsOpt(canonicalToFile, canonicalToBase);
            if (canonicalRelative != null) {
                return canonicalRelative;
            }
        }
        catch (IOException ignored) {
            Exceptions.ignore(ignored, "Fall through to comparing standard paths");
        }
        String relative = FileUtil.relativePathAsIsOpt(file, baseDirectory);
        return relative;
    }

    public static String relativePathAsIsOpt(File childFile, File parentFile) {
        String child = childFile.getPath();
        String parent = parentFile.getPath();
        int parentLength = parent.length();
        if (child.length() <= parentLength) {
            return null;
        }
        if (!child.startsWith(parent)) {
            return null;
        }
        if (child.charAt(parentLength) == File.separatorChar) {
            return child.substring(parentLength + 1);
        }
        if (parentLength > 0 && child.charAt(parentLength - 1) == File.separatorChar) {
            return child.substring(parentLength);
        }
        return null;
    }

    public static boolean isWithin(File file, File baseDirectory) {
        return FileUtil.relativePathOpt(file, baseDirectory) != null || FileUtil.tryMakeCanonical(file).equals(FileUtil.tryMakeCanonical(baseDirectory));
    }

    public static File tryMakeCanonical(File f) {
        try {
            return f.getCanonicalFile();
        }
        catch (IOException ignored) {
            Exceptions.ignore(ignored, "Can't log error: Could be too verbose.");
            return new File(FileUtil.simplifyPath(f));
        }
    }

    public static boolean recursiveParentOf(File parent, File child) {
        return FileUtil.relativePathOpt(child, parent) != null;
    }

    @Deprecated
    public static boolean recursiveDelete(File file) {
        return FileUtil.recursiveDelete(file, falseFilter) == DeletionResult.Deleted;
    }

    @Deprecated
    public static DeletionResult recursiveDelete(File file, FileFilter exceptions) {
        if (exceptions.accept(file)) {
            return DeletionResult.SkippedSomeFiles;
        }
        if (file.delete()) {
            return DeletionResult.Deleted;
        }
        if (file.isDirectory()) {
            File[] children = file.listFiles();
            if (children == null) {
                return DeletionResult.Failed;
            }
            boolean skippedSomeChildren = false;
            for (File f : children) {
                DeletionResult childResult = FileUtil.recursiveDelete(f, exceptions);
                if (childResult == DeletionResult.Failed) {
                    return DeletionResult.Failed;
                }
                if (childResult != DeletionResult.SkippedSomeFiles) continue;
                skippedSomeChildren = true;
            }
            if (file.delete()) {
                return DeletionResult.Deleted;
            }
            return skippedSomeChildren ? DeletionResult.SkippedSomeFiles : DeletionResult.Failed;
        }
        return DeletionResult.Failed;
    }

    public static String simplifyPath(File file) {
        return file.toPath().toAbsolutePath().normalize().toString();
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static Map<String, String> readPropertiesCSV(File file) {
        LinkedHashMap<String, String> result = new LinkedHashMap<String, String>();
        try (BufferedReader reader = Files.newBufferedReader(file.toPath(), UTF8);){
            Map map;
            try (CSVReader csv = new CSVReader(reader);){
                for (String[] line : csv.readAll()) {
                    if (line.length < 1) continue;
                    String key = line[0];
                    String value = null;
                    if (line.length >= 2) {
                        value = line[1];
                    }
                    result.put(key, value);
                }
                map = Collections.unmodifiableMap(result);
            }
            return map;
        }
        catch (UnsupportedEncodingException e) {
            throw new CatastrophicError(e);
        }
        catch (IOException e) {
            throw new ResourceError("Could not read data from " + file, e);
        }
    }

    public static Writer openWriterUTF8(File file, boolean overwrite, boolean append) {
        try {
            OpenOption[] options = append ? new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.APPEND} : (overwrite ? new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING} : new OpenOption[]{StandardOpenOption.CREATE_NEW});
            return Files.newBufferedWriter(file.toPath(), UTF8, options);
        }
        catch (FileAlreadyExistsException faee) {
            throw new ResourceError("Could not delete existing file: " + file, faee);
        }
        catch (IOException ioe) {
            throw new ResourceError("Failed to open FileWriter for " + file, ioe);
        }
    }

    public static boolean isReadableFile(File path) {
        return path != null && path.isFile() && path.canRead();
    }

    public static boolean isSamePath(File path1, File path2) {
        if (path1 == null || path2 == null) {
            return false;
        }
        return ObjectUtil.equals(FileUtil.tryMakeCanonical(path1), FileUtil.tryMakeCanonical(path2));
    }

    public static File addExtension(File file, String extension) {
        return new File(file.getPath() + "." + extension);
    }

    public static File createTempDir() {
        File globalTempDir = new File(System.getProperty("java.io.tmpdir"));
        return FileUtil.createUniqueDirectory(globalTempDir, "semmleTempDir");
    }

    private static void performWithRetries(RetryablePathConsumer operation, Path source, Path target, BiFunction<Path, Path, String> errorMessageCreator) throws IOException {
        int i = 1;
        while (true) {
            try {
                operation.accept(source, target);
                return;
            }
            catch (AtomicMoveNotSupportedException e) {
                throw e;
            }
            catch (IOException e) {
                String message = errorMessageCreator.apply(source, target);
                logger.warn(message + " (attempt " + i + ")", e);
                if (i == 30) {
                    throw new IOException(message + "(" + 30 + " attempts made)", e);
                }
                try {
                    logger.trace("Waiting for 1000 ms before making another attempt.");
                    Thread.sleep(1000L);
                }
                catch (InterruptedException e2) {
                    logger.warn("Thread interrupted before making another attempt.", e2);
                }
                ++i;
                continue;
            }
            break;
        }
    }

    public static void moveWithRetries(Path source, Path target, CopyOption ... options) throws IOException {
        FileUtil.performWithRetries((s, t) -> Files.move(s, t, options), source, target, (s, t) -> "Failed to perform move from " + s.toAbsolutePath() + " to " + t.toAbsolutePath());
    }

    static {
        int i;
        logger = LoggerFactory.getLogger(FileUtil.class);
        rpInvalidFilenameCharacters = Pattern.compile("[\\\\/:*?\"'<>|@]");
        UTF8 = StandardCharsets.UTF_8;
        allowedCharacters = new BitSet(256);
        for (i = 97; i <= 122; ++i) {
            allowedCharacters.set(i);
        }
        for (i = 65; i <= 90; ++i) {
            allowedCharacters.set(i);
        }
        for (i = 48; i <= 57; ++i) {
            allowedCharacters.set(i);
        }
        allowedCharacters.set(45);
        allowedCharacters.set(95);
        allowedCharacters.set(32);
        allowedCharacters.set(91);
        allowedCharacters.set(93);
        falseFilter = new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return false;
            }
        };
    }

    private static class FileFilterImpl
    implements FileFilter {
        private final String[] names;
        private final boolean caseSensitive;
        private final boolean extensionOnly;

        private FileFilterImpl(String[] names, boolean caseSensitive, boolean extensionOnly) {
            this.names = Arrays.copyOf(names, names.length);
            this.caseSensitive = caseSensitive;
            this.extensionOnly = extensionOnly;
            if (!caseSensitive) {
                for (int i = 0; i < this.names.length; ++i) {
                    this.names[i] = StringUtil.lc(this.names[i]);
                }
            }
        }

        @Override
        public boolean accept(File pathname) {
            if (!pathname.isFile()) {
                return false;
            }
            String nameToMatch = this.caseSensitive ? pathname.getName() : StringUtil.lc(pathname.getName());
            for (String s : this.names) {
                if (this.extensionOnly && nameToMatch.endsWith(s)) {
                    return true;
                }
                if (this.extensionOnly || !nameToMatch.equals(s)) continue;
                return true;
            }
            return false;
        }
    }

    public static class ResolvedCompressedSourceArchivePaths {
        public final String srcArchivePath;
        public final String sourceLocationPrefix;

        public ResolvedCompressedSourceArchivePaths(String srcArchivePath, String sourceLocationPrefix) {
            this.srcArchivePath = srcArchivePath;
            this.sourceLocationPrefix = sourceLocationPrefix;
        }
    }

    public static enum DeletionResult {
        Deleted,
        SkippedSomeFiles,
        Failed;

    }

    private static interface RetryablePathConsumer {
        public void accept(Path var1, Path var2) throws IOException;
    }
}

