/*
 * Decompiled with CFR 0.152.
 */
package com.semmle.util.trap;

import com.semmle.util.data.Date;
import com.semmle.util.data.LRUCache;
import com.semmle.util.data.StringUtil;
import com.semmle.util.exception.CatastrophicError;
import com.semmle.util.exception.ResourceError;
import com.semmle.util.files.FileUtil;
import com.semmle.util.trap.TrapDataType;
import com.semmle.util.trap.pathtransformers.PathTransformer;
import com.semmle.util.unicode.UTF8Util;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;
import org.slf4j.Logger;
import org.slf4j.Marker;

public class TrapWriter
implements Closeable {
    private static final int MAX_STRLEN = 0x100000;
    private static final Charset UTF8 = Charset.forName("UTF-8");
    protected final Writer out;
    protected final File trapFile;
    protected final File tmpFile;
    private final Logger log;
    private final Marker logId;
    private int idcount = 10000;
    private final Map<File, Label> fileAndFolderLabels = new LRUCache<File, Label>(20000);
    private final Map<String, Label> labelFor = new LRUCache<String, Label>(20000);
    private final Map<LabelDefinition, Label> localLabels = new LinkedHashMap<LabelDefinition, Label>();
    private final boolean concurrentCreation;
    private final PathTransformer transformer;
    private static final Pattern doubleQuote = Pattern.compile("\"", 16);
    private static final String[] keyReplacementMap = new String[127];

    public TrapWriter(File trapFile) {
        this(trapFile, false);
    }

    public TrapWriter(Writer out, PathTransformer transformer) {
        this(out, transformer, null, null);
    }

    private TrapWriter(Writer out, PathTransformer transformer, Logger log, Marker logId) {
        this.log = log;
        this.logId = logId;
        this.trapFile = null;
        this.concurrentCreation = false;
        this.tmpFile = null;
        this.out = out;
        this.transformer = transformer;
    }

    public static TrapWriter concurrentWriter(File trapFile) {
        return TrapWriter.concurrentWriter(trapFile, null, null);
    }

    public static TrapWriter concurrentWriter(File trapFile, Logger log, Marker logId) {
        if (trapFile.exists()) {
            return null;
        }
        return new TrapWriter(trapFile, true, log, logId);
    }

    protected TrapWriter(File trapFile, boolean concurrentCreation) {
        this(trapFile, concurrentCreation, null, null);
    }

    protected TrapWriter(File trapFile, boolean concurrentCreation, Logger log, Marker logId) {
        this.log = log;
        this.logId = logId;
        this.trapFile = trapFile;
        this.concurrentCreation = concurrentCreation;
        if (!StringUtil.lc(trapFile.getName()).endsWith(".gz")) {
            throw new CatastrophicError("TrapWriters will produce gzipped files and should use the .gz extension");
        }
        FileOutputStream outputStream = null;
        BufferedOutputStream bufferedStream = null;
        GZIPOutputStream gzipOutputStream = null;
        try {
            if (concurrentCreation) {
                this.tmpFile = FileUtil.createUniqueFile(trapFile.getParentFile(), trapFile.getName() + ".tmp");
                outputStream = new FileOutputStream(this.tmpFile);
            } else {
                outputStream = new FileOutputStream(trapFile);
                this.tmpFile = null;
            }
            bufferedStream = new BufferedOutputStream(outputStream);
            gzipOutputStream = new GZIPOutputStream(bufferedStream);
        }
        catch (IOException e) {
            FileUtil.close(gzipOutputStream);
            FileUtil.close(bufferedStream);
            FileUtil.close(outputStream);
            throw new ResourceError("Failed to created the parent directory for output file " + trapFile + ", or could not write to this directory", e);
        }
        CharsetEncoder encoder = UTF8.newEncoder().replaceWith("\ufffd".getBytes(UTF8)).onMalformedInput(CodingErrorAction.REPLACE).onUnmappableCharacter(CodingErrorAction.REPLACE);
        this.out = new BufferedWriter(new OutputStreamWriter((OutputStream)gzipOutputStream, encoder));
        this.transformer = PathTransformer.std();
    }

    private String escapeString(String s) {
        return doubleQuote.matcher(s).replaceAll("\"\"");
    }

    public static String escapeKey(String s) {
        StringBuilder sb = null;
        int lastIndex = 0;
        for (int i = 0; i < s.length(); ++i) {
            char ch = s.charAt(i);
            switch (ch) {
                case '\"': 
                case '#': 
                case '&': 
                case '@': 
                case '{': 
                case '}': {
                    if (sb == null) {
                        sb = new StringBuilder();
                    }
                    sb.append(s, lastIndex, i);
                    sb.append(keyReplacementMap[ch]);
                    lastIndex = i + 1;
                }
            }
        }
        if (sb != null) {
            sb.append(s, lastIndex, s.length());
            return sb.toString();
        }
        return s;
    }

    private String nextLabel() {
        String id = "#" + this.idcount++;
        return id;
    }

    public boolean bumpIdCount(int idcount) {
        if (this.idcount > idcount) {
            return false;
        }
        this.idcount = idcount;
        return true;
    }

    public Label freshLabel() {
        String id = this.nextLabel();
        this.print(id);
        this.print("=*\n");
        return new Label(id);
    }

    public Label globalID(String key) {
        Label result = this.labelFor.get(key);
        if (result == null) {
            String id = this.nextLabel();
            this.print(id);
            this.print("=@\"");
            this.print(this.escapeString(key));
            this.print("\"\n");
            result = new Label(id);
            this.labelFor.put(key, result);
        }
        return result;
    }

    public Label localID(Object key) {
        LabelDefinition labeldef = new LabelDefinition(key);
        Label result = this.localLabels.get(labeldef);
        if (result == null) {
            result = this.freshLabel();
            this.localLabels.put(labeldef, result);
        }
        return result;
    }

    public Label localID(Object key, String kind) {
        LabelDefinition labeldef = new LabelDefinition(key, kind);
        Label result = this.localLabels.get(labeldef);
        if (result == null) {
            result = this.freshLabel();
            this.localLabels.put(labeldef, result);
        }
        return result;
    }

    public void clearLocalIDs() {
        this.localLabels.clear();
    }

    public void addTuple(Table table, Object ... values) {
        if (values.length != table.getArity()) {
            throw new CatastrophicError("Trying to insert a tuple of size " + values.length + " into table " + table.getName() + " of arity " + table.getArity());
        }
        if (table.validate(values)) {
            this.addTuple(table.getName(), values);
        } else if (this.log != null) {
            this.log.warn(this.logId, "Skipping invalid tuple: " + table.getName() + "(" + StringUtil.glue(",", values) + ") in file " + this.trapFile.getAbsolutePath());
        }
    }

    public void addTuple(String tableName, Object ... values) {
        StringBuilder sb = new StringBuilder();
        sb.append(tableName);
        sb.append("(");
        boolean first = true;
        for (Object value : values) {
            if (!first) {
                sb.append(",");
            } else {
                first = false;
            }
            if (value instanceof Label) {
                sb.append(((Label)value).toString());
                continue;
            }
            if (value instanceof Date) {
                sb.append("D\"");
                sb.append(((Date)value).toString());
                sb.append("\"");
                continue;
            }
            if (value instanceof String) {
                String strval = (String)value;
                strval = strval.substring(0, UTF8Util.encodablePrefixLength(strval, 0x100000));
                sb.append("\"");
                sb.append(this.escapeString(strval));
                sb.append("\"");
                continue;
            }
            if (value instanceof Number) {
                sb.append(value.toString());
                continue;
            }
            if (value instanceof Boolean) {
                sb.append(value.toString());
                continue;
            }
            if (value instanceof TrapDataType) {
                sb.append(((TrapDataType)value).toTrapString());
                continue;
            }
            if (value == null) {
                throw new CatastrophicError("Trying to insert a tuple containing NULL in table " + tableName + ": (" + StringUtil.glue(",", values) + ")");
            }
            throw new RuntimeException("Trying to insert tuple of unknown kind: " + value.getClass().getName());
        }
        sb.append(")\n");
        this.print(sb.toString());
    }

    public void writeComment(String comment) {
        this.print("// " + comment.replace("\n", "\n//    ") + "\n");
    }

    private void print(String s) {
        try {
            this.out.write(s);
        }
        catch (IOException e) {
            throw new ResourceError("Error trying to emit tuple to trap file", e);
        }
    }

    public Label populateUnknownFile() {
        Label result = this.fileAndFolderLabels.get(new File(""));
        if (result == null) {
            result = this.globalID("unknown;sourcefile");
            this.addTuple("files", result, "");
            this.fileAndFolderLabels.put(new File(""), result);
        }
        return result;
    }

    public Label populateFile(File absoluteFile) {
        Label result = this.fileAndFolderLabels.get(absoluteFile);
        if (result == null) {
            String databasePath = this.transformer.fileAsDatabaseString(absoluteFile);
            File normalisedFile = new File(databasePath);
            result = this.globalID(TrapWriter.escapeKey(databasePath) + ";sourcefile");
            this.addTuple("files", result, databasePath);
            this.populateParents(normalisedFile, result);
            this.fileAndFolderLabels.put(absoluteFile, result);
        }
        return result;
    }

    public Label populateFolder(File folder) {
        Label result = this.fileAndFolderLabels.get(folder);
        if (result == null) {
            String databasePath = this.transformer.fileAsDatabaseString(folder);
            File normalisedFolder = new File(databasePath);
            result = this.addFolderTuple(databasePath);
            this.populateParents(normalisedFolder, result);
            this.fileAndFolderLabels.put(folder, result);
        }
        return result;
    }

    public Label location(Label file, int startLine, int startColumn, int endLine, int endColumn) {
        return this.globalID("loc,{" + file + "}," + startLine + "," + startColumn + "," + endLine + "," + endColumn);
    }

    private Label addFolderTuple(String databasePath) {
        Label result = this.globalID(TrapWriter.escapeKey(databasePath) + ";folder");
        this.addTuple("folders", result, databasePath);
        return result;
    }

    private void populateParents(File normalisedFile, Label label) {
        File parent = normalisedFile.getParentFile();
        if (parent == null) {
            return;
        }
        Label parentLabel = this.fileAndFolderLabels.get(parent);
        if (parentLabel == null) {
            parentLabel = this.addFolderTuple(FileUtil.normalisePath(parent.getPath()));
            this.populateParents(parent, parentLabel);
            this.fileAndFolderLabels.put(parent, parentLabel);
        }
        this.addTuple("containerparent", parentLabel, label);
    }

    @Override
    public void close() {
        FileUtil.close(this.out);
        if (this.concurrentCreation) {
            if (!this.tmpFile.renameTo(this.trapFile) && !this.trapFile.exists()) {
                throw new ResourceError("Couldn't rename " + this.tmpFile + " to " + this.trapFile);
            }
            this.tmpFile.delete();
        }
    }

    static {
        TrapWriter.keyReplacementMap[38] = "&amp;";
        TrapWriter.keyReplacementMap[123] = "&lbrace;";
        TrapWriter.keyReplacementMap[125] = "&rbrace;";
        TrapWriter.keyReplacementMap[34] = "&quot;";
        TrapWriter.keyReplacementMap[64] = "&commat;";
        TrapWriter.keyReplacementMap[35] = "&num;";
    }

    public static class Label {
        private final String id;

        private Label(String id) {
            this.id = id;
        }

        public String toString() {
            return this.id;
        }
    }

    private static class LabelDefinition {
        private final Object o;
        private final String kind;

        LabelDefinition(Object o) {
            this.o = o;
            this.kind = null;
        }

        LabelDefinition(Object o, String kind) {
            this.o = o;
            this.kind = kind;
        }

        public int hashCode() {
            return System.identityHashCode(this.o) + (this.kind == null ? 0 : this.kind.hashCode());
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (obj.getClass() != this.getClass()) {
                return false;
            }
            LabelDefinition that = (LabelDefinition)obj;
            if (this.kind == null) {
                return that.kind == null && this.o == that.o;
            }
            return this.kind.equals(that.kind) && this.o == that.o;
        }
    }

    public static interface Table {
        public String getName();

        public int getArity();

        public boolean validate(Object ... var1);
    }
}

