
import __builtin__
import imp
import keyword
import sys
import traceback
import importlib
import os.path
from types import ModuleType
from functools import total_ordering
DEFAULT_TYPE_MAP = {8: 'bool', 12: 'bytes', 14: 'int', 7: 'int', 6: 'int', 15: 'int', 16: 'int', 2: 'float', 1: 'float', 9: 'unicode', 5: 'int', 3: 'int', 17: 'int', 18: 'int'}
TYPE_MAP = None

def init_type_map(descriptor):
    global TYPE_MAP
    if (TYPE_MAP is not None):
        return
    try:
        with _CustomImportHooks([]):
            module = importlib.import_module(type(descriptor).__module__)
        FieldDescriptor = getattr(module, 'FieldDescriptor')
    except Exception as ex:
        print(('Falling back on default type-map for protobuf: ' + str(ex)))
        print(('Descriptor: ' + str(descriptor)))
        TYPE_MAP = DEFAULT_TYPE_MAP
        return
    TYPE_MAP = {FieldDescriptor.TYPE_BOOL: 'bool', FieldDescriptor.TYPE_BYTES: 'bytes', FieldDescriptor.TYPE_ENUM: 'int', FieldDescriptor.TYPE_FIXED32: 'int', FieldDescriptor.TYPE_FIXED64: 'int', FieldDescriptor.TYPE_SFIXED32: 'int', FieldDescriptor.TYPE_SFIXED64: 'int', FieldDescriptor.TYPE_FLOAT: 'float', FieldDescriptor.TYPE_DOUBLE: 'float', FieldDescriptor.TYPE_STRING: 'unicode', FieldDescriptor.TYPE_INT32: 'int', FieldDescriptor.TYPE_INT64: 'int', FieldDescriptor.TYPE_UINT32: 'int', FieldDescriptor.TYPE_UINT64: 'int', FieldDescriptor.TYPE_SINT32: 'int', FieldDescriptor.TYPE_SINT64: 'int'}

class _DefaultFinder(object):
    disable_custom_finders = False

    def __init__(self):
        self.cache = {}

    def find_module(self, fullname, path=None):
        if _DefaultFinder.disable_custom_finders:
            return None
        _DefaultFinder.disable_custom_finders = True
        try:
            self.cache[fullname] = importlib.import_module(fullname)
        except ImportError:
            return None
        finally:
            _DefaultFinder.disable_custom_finders = False
        return self

    def load_module(self, fullname):
        return self.cache[fullname]

class _NoInitFinder(object):

    def __init__(self, extractor_path, fake_modules=False):
        self.extractor_path = extractor_path
        self.fake_modules = fake_modules

    def all_paths(self):
        for path in self.extractor_path:
            (yield path)
        for path in sys.path:
            (yield path)

    def find_module(self, fullname, path=None):
        if _DefaultFinder.disable_custom_finders:
            return None
        if ((fullname in sys.modules) or (fullname.split('/')[(- 1)] in sys.modules)):
            return None
        if (imp.is_builtin(fullname) or imp.is_frozen(fullname) or fullname.endswith('/sys') or (fullname == 'sys')):
            return None
        for element in self.all_paths():
            fullpath = os.path.join(element, *fullname.split('.'))
            nested_init = os.path.join(fullpath, '__init__.py')
            py_path = (fullpath + '.py')
            if (os.path.isfile(nested_init) or os.path.isfile(py_path)):
                return None
            if os.path.isdir(fullpath):
                return self
        if self.fake_modules:
            return self
        return None

    def load_module(self, fullname):
        if (fullname in sys.modules):
            return sys.modules[fullname]
        mod = None
        for element in self.all_paths():
            fullpath = os.path.join(element, *fullname.split('.'))
            if (not os.path.isdir(fullpath)):
                continue
            mod = ModuleType(fullname)
            break
        else:
            if (not self.fake_modules):
                raise ImportError
            mod = AnyModule(fullname)
        sys.modules[fullname] = mod
        mod.__path__ = [fullpath]
        mod.__package__ = fullname
        mod.__loader__ = self
        sys.stdout.flush()
        sys.stderr.flush()
        return mod

@total_ordering
class AnyObject(object):

    def __init__(self, *args, **kwargs):
        pass

    def __add__(self, o):
        if isinstance(o, basestring):
            return (str(self) + o)
        return AnyObject()

    def __radd__(self, o):
        if isinstance(o, basestring):
            return (o + str(self))
        return AnyObject()

    def __getattribute__(self, name):
        try:
            return object.__getattribute__(self, name)
        except AttributeError:
            return AnyObject()

    def __getitem__(self, index):
        return AnyObject()

    def __setitem__(self, index, value):
        return None

    def __call__(self, *args, **kwargs):
        return AnyObject()

    def __lt__(self, other):
        return False

    def __nonzero__(self):
        return False

    def __bool__(self):
        return False

    def __get__(self, instance, owner):
        return AnyObject()

    def __set__(self, instance, value):
        return None

    def __delete__(self, instance):
        return None

    def __str__(self):
        return ('unknown_%d______' % id(self))

class AnyNamedObject(AnyObject):

    def __init__(self, name, *args, **kwargs):
        self.__name__ = name

    def __str__(self):
        return self.__name__

    def __getattribute__(self, name):
        try:
            return object.__getattribute__(self, name)
        except AttributeError:
            return AnyNamedObject(((object.__getattribute__(self, '__name__') + '.') + name))

class AnyModule(AnyNamedObject):

    def __init__(self, fullname):
        super(AnyModule, self).__init__(fullname)

    @property
    def name(self):
        return self.__name__

class _CustomImportHooks(object):
    builtin_import = __import__
    stub_sys_modules = {}

    def __init__(self, extractor_path):
        self.extractor_path = extractor_path
        self.sys_modules_orig = None
        self.sys_path_orig = None
        self.sys_meta_path_orig = None

    def __enter__(self):
        self.sys_modules_orig = sys.modules.copy()
        self.sys_path_orig = list(sys.path)
        self.sys_meta_path_orig = list(sys.meta_path)
        sys.modules.update(_CustomImportHooks.stub_sys_modules)
        sys.path += self.extractor_path
        sys.meta_path.append(_DefaultFinder())
        sys.meta_path.append(_NoInitFinder(self.extractor_path, True))
        __builtin__.__import__ = _CustomImportHooks.custom_import
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        _CustomImportHooks.stub_sys_modules.update(sys.modules)
        sys.modules.clear()
        sys.modules.update(self.sys_modules_orig)
        sys.path[:] = []
        sys.path += self.sys_path_orig
        sys.meta_path[:] = []
        sys.meta_path += self.sys_meta_path_orig
        __builtin__.__import__ = _CustomImportHooks.builtin_import
        return False

    @staticmethod
    def custom_import(name, globals=None, locals=None, fromlist=None, level=(- 1)):
        if (globals is None):
            globals = {}
        if (locals is None):
            locals = {}
        if (fromlist is None):
            fromlist = []
        try:
            return _CustomImportHooks.builtin_import(name, globals, locals, fromlist, level)
        except ImportError as e:
            raise
        except Exception as e:
            print('Proto stubbing: Suppressing {} with AnyObject for {}.'.format(e, name))
        return AnyModule(name)

def eval_descriptor(src, extractor_path):
    try:
        exec_globals = {}
        with _CustomImportHooks(extractor_path):
            exec(src, exec_globals, exec_globals)
        descriptor = exec_globals['DESCRIPTOR']
    except ImportError as ie:
        extra_message = ''
        try:
            extra_message += (' | sys.path: ' + ', '.join(sys.path))
            extra_message += (' | extractor.path: ' + ', '.join(extractor_path))
            possibles = []
            module_name = ie.message[len('No module found '):]
            for element in (extractor_path + sys.path):
                fullpath = os.path.join(element, *module_name.split('.'))
                for modpath in (fullpath, (fullpath + '.py'), (fullpath + '.pyc')):
                    if os.path.exists(modpath):
                        possibles.append(modpath)
            if possibles:
                extra_message += (' | Possible matches: ' + ', '.join(possibles))
            else:
                extra_message += ' | No matches'
        finally:
            ie.args = ((', '.join(ie.args) + extra_message),)
            raise ie
    except Exception as e:
        print('Proto stubbing: {}'.format(e))
        traceback.print_exc()
        raise
    if isinstance(descriptor, AnyObject):
        raise ImportError('Giving up on module with unresolved DESCRIPTOR {}.'.format(descriptor))
    init_type_map(descriptor)
    return (exec_globals, descriptor)

class StubModule(object):

    def __init__(self, src, extractor_path):
        self.imports = set()
        self.classes = []
        (exec_globals, descriptor) = eval_descriptor(src, extractor_path)
        self.module = module_name_from_file_descriptor(descriptor)
        for (name, item) in sorted(exec_globals.items()):
            if (isinstance(item, type) and hasattr(item, 'DESCRIPTOR')):
                self.classes.append(StubClass(item, self))

    def get_source(self):
        out = []
        for (module, cls) in sorted(self.imports):
            if (isinstance(module, AnyObject) or isinstance(cls, AnyObject)):
                continue
            out.append(('from %s import %s\n' % (module, cls.split('.')[0])).replace('-', '_'))
        out.append('\n')
        for cls in self.classes:
            out.append(cls.get_source(''))
        return ''.join(out)

class StubClass(object):

    def __init__(self, proto_cls, module):
        self.classes = []
        self.enums = []
        self.attrs = []
        self.methods = []
        self.name = proto_cls.__name__
        self.qualified_name = class_name(proto_cls.DESCRIPTOR)
        supercls = proto_cls.mro()[1]
        superclassname = supercls.__name__
        self.super_name = superclassname
        module.imports.add((supercls.__module__, superclassname))
        descriptor = proto_cls.DESCRIPTOR
        if (type(descriptor).__name__ == 'ServiceDescriptor'):
            self.is_service = True
            for method in descriptor.methods:
                self.methods.append((method.name, class_name(method.input_type), class_name(method.output_type)))
                add_to_imports(method.input_type, module)
                add_to_imports(method.output_type, module)
            return
        self.is_service = False
        if (not hasattr(descriptor, 'fields_by_name')):
            print(('Proto stubbing: Unexpectedly missing fields_by_name attribute for {}, ' + 'present is {}. Descriptor name: {}, methods: {}').format(proto_cls, dir(descriptor), descriptor.name, descriptor.methods))
            raise TypeError()
        fields = descriptor.fields_by_name
        for (name, value) in sorted(fields.items()):
            (classname, is_list) = class_from_field_descriptor(value, module)
            self.attrs.append((name, classname, is_list))
        for (name, val_desc) in descriptor.enum_values_by_name.items():
            self.enums.append((name, val_desc.number))
        for item in proto_cls.__dict__.values():
            if (isinstance(item, type) and hasattr(item, 'DESCRIPTOR') and isinstance(item, type)):
                self.classes.append(StubClass(item, module))

    def get_source(self, indent):
        out = []
        out.append(('%sclass %s(%s):\n' % (indent, self.name, self.super_name)))
        out.append('\n')
        out.append((indent + '    def __init__(self):\n'))
        if self.attrs:
            for (name, cls, repeated) in self.attrs:
                if isinstance(cls, AnyObject):
                    out.append((('# Unknown type ' + str(cls).replace('\n', '')) + '\n'))
                    cls = 'unknown'
                if keyword.iskeyword(name):
                    template = '%s        setattr(self, "%s", %s)\n'
                else:
                    template = '%s        self.%s = %s\n'
                if repeated:
                    value = ('[%s()]' % cls)
                else:
                    value = ('%s()' % cls)
                out.append((template % (indent, name, value)))
        else:
            out.append((indent + '        pass\n'))
        out.append('\n')
        if self.is_service:
            out.append((indent + '    def GetRequestClass(self, method_descriptor):\n'))
            out.append((indent + '        unknown()\n\n'))
            out.append((indent + '    def GetResponseClass(self, method_descriptor):\n'))
            out.append((indent + '        unknown()\n\n'))
        else:
            out.append((indent + '    def add(self):\n'))
            out.append(('%s        return %s()\n' % (indent, self.qualified_name)))
            out.append('\n')
            out.append((indent + '    def ParseFromString(self, txt):\n'))
            out.append((indent + '        pass\n'))
            out.append('\n')
            out.append((indent + '    def SerializeToString(self):\n'))
            out.append((indent + '        return str()\n'))
            out.append('\n')
        for (name, val) in self.enums:
            out.append(('%s    %s = %s\n' % (indent, name, val)))
        if self.enums:
            out.append('\n')
        for (name, in_type, out_type) in self.methods:
            out.append(('%s    def %s(self, rpc_controller, request, done):\n' % (indent, name)))
            out.append((indent + ('        assert isinstance(request, %s)\n' % in_type)))
            out.append((indent + ('        done(%s())\n\n' % out_type)))
        for cls in self.classes:
            out.append(cls.get_source((indent + '    ')))
        return ''.join(out)

def module_name_from_file_descriptor(file_descriptor):
    return (('google3.' + str(file_descriptor.name)[:(- 6)].replace('/', '.')) + '_pb2')

def module_name(descriptor):
    return module_name_from_file_descriptor(descriptor.file)

def class_name(descriptor):
    if descriptor.containing_type:
        return ((class_name(descriptor.containing_type) + '.') + descriptor.name)
    else:
        return descriptor.name

def add_to_imports(descriptor, module):
    modulename = module_name(descriptor)
    cls = class_name(descriptor)
    if (modulename != module.module):
        module.imports.add((modulename, cls))

def class_from_field_descriptor(descriptor, module):
    message_type = descriptor.message_type
    if (message_type is not None):
        add_to_imports(message_type, module)
        cls = class_name(message_type)
        is_list = False
    elif (descriptor.type in TYPE_MAP):
        cls = TYPE_MAP[descriptor.type]
    else:
        cls = ('Type%d' % descriptor.type)
    is_list = is_repeated_field(descriptor)
    return (cls, is_list)

def is_repeated_field(descriptor):
    return (descriptor.label == descriptor.LABEL_REPEATED)

def stub_source(original_source, extractor_path):
    stub_src = StubModule(original_source, extractor_path).get_source()
    return stub_src

def main():
    for file in sys.argv[1:]:
        if file.endswith('.py'):
            print(stub_source(open(file).read(), []))
if (__name__ == '__main__'):
    main()
