
import os.path
import sys
from semmle.util import base64digest, str_to_unicode, unicode_to_str, makedirs
_WINDOWS = (os.name == 'nt')
LONG_PATH_PREFIX = '\\\\?\\'

def make_renamer(renamer):
    if (os.name == 'nt'):
        if (renamer is None):
            return (lambda path: path.replace('\\', '/'))
        else:
            return (lambda path: renamer(path).replace('\\', '/'))
    elif (renamer is None):
        return (lambda path: path)
    else:
        return renamer

class NullArchive(object):

    def __init__(self, renamer=None):
        self.renamer = make_renamer(renamer)

    def write(self, path, source, encoding=None):
        pass

    def get_virtual_path(self, real_path):
        return self.renamer(real_path)

class RenamingFolder(object):

    def __init__(self, folder, renamer, logger):
        assert (folder is not None)
        self.folder = folder
        self.renamer = make_renamer(renamer)
        self.logger = logger

    def get_virtual_path(self, real_path):
        return self.renamer(real_path)

    def get_storage_path(self, *subpath):
        subpath = [p for p in subpath if p]
        suffix = os.sep.join(subpath)
        suffix = suffix.replace(':', '_')
        if (suffix[0] in '/\\'):
            result = (self.folder + suffix)
        else:
            result = ((self.folder + os.sep) + suffix)
        if (_WINDOWS and (len(result) > 240)):
            result = (LONG_PATH_PREFIX + result)
        folder = os.path.dirname(result)
        if (not os.path.exists(folder)):
            makedirs(folder)
        return result

class TrapFolder(RenamingFolder):

    def _trap_path(self, namespace, path, extension='.trap.gz'):
        vpath = self.get_virtual_path(path)
        parts = vpath.split('/')
        basename = parts[(- 1)]
        hashcode = base64digest((str_to_unicode(vpath) + namespace))
        filename = (((basename + '.') + unicode_to_str(hashcode, sys.getfilesystemencoding())) + extension)
        return self.get_storage_path(filename)

    def write_trap(self, namespace, path, data, extension='.trap.gz'):
        outpath = self._trap_path(namespace, path, extension)
        with open(outpath, 'wb') as out:
            out.write(data)

class SourceArchive(RenamingFolder):

    def write(self, path, bytes_source):
        vpath = self.get_virtual_path(path)
        if (vpath != path):
            self.logger.debug("Renaming '%s' to '%s'", path, vpath)
        self.logger.debug("Writing source to '%s'", vpath)
        subpath = vpath.split('/')
        outpath = self.get_storage_path(*subpath)
        with open(outpath, 'wb') as out:
            out.write(bytes_source)
