
from __future__ import absolute_import
import os
import sys
import traceback
import multiprocessing
OFF = 0
ERROR = 1
WARN = 2
INFO = 3
DEBUG = 4
TRACE = 5
TRACEBACK = 6
COLOR = 8
if (os.name == 'nt'):
    MAGENTA = ''
    GREY = ''
    BLUE = ''
    YELLOW = ''
    RED = ''
    RESET = ''
else:
    MAGENTA = '\x1b[35m'
    GREY = '\x1b[2m\x1b[37m'
    BLUE = '\x1b[34m'
    YELLOW = '\x1b[33m'
    RED = '\x1b[31m'
    RESET = '\x1b[0m'
LOG_PREFIX = {TRACE: '[TRACE] ', DEBUG: '[DEBUG] ', INFO: '[INFO] ', WARN: '[WARN] ', ERROR: '[ERROR] ', TRACEBACK: '[TRACEBACK] ', (COLOR | TRACE): (GREY + '[TRACE] '), (COLOR | DEBUG): '[DEBUG] ', (COLOR | INFO): (BLUE + '[INFO] '), (COLOR | WARN): (YELLOW + '[WARN] '), (COLOR | ERROR): (RED + '[ERROR] '), (COLOR | TRACEBACK): (MAGENTA + '[TRACEBACK] ')}

def write_message(level, text):
    reset = (RESET if ((level & COLOR) == COLOR) else '')
    print(((LOG_PREFIX[level] + text) + reset))
    sys.stdout.flush()

def write_message_with_proc(level, proc_id, text):
    reset = (RESET if ((level & COLOR) == COLOR) else '')
    print((((LOG_PREFIX[level] + proc_id) + text) + reset))
    sys.stdout.flush()
_logging_process = None

def stop():
    _logging_process.join()

class Logger(object):

    def __init__(self, level=WARN, color=False):
        global _logging_process
        self.proc_id = ''
        self.level = level
        method = ('spawn' if (sys.platform == 'darwin') else None)
        try:
            ctx = multiprocessing.get_context(method)
        except AttributeError:
            ctx = multiprocessing
        self.queue = ctx.Queue()
        _logging_process = ctx.Process(target=_message_loop, args=(self.queue,))
        _logging_process.start()
        self.color = (COLOR if color else 0)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()
        return False

    def set_process_id(self, proc_id):
        self.proc_id = ('[%d] ' % proc_id)

    def setLevel(self, level):
        self.level = level

    def log(self, level, fmt, *args):
        if (level <= self.level):
            txt = (fmt % args)
            try:
                self.queue.put(((self.color | level), self.proc_id, txt), False)
            except Exception:
                self.write_message((self.color | level), txt)

    def debug(self, fmt, *args):
        self.log(DEBUG, fmt, *args)

    def info(self, fmt, *args):
        self.log(INFO, fmt, *args)

    def warning(self, fmt, *args):
        self.log(WARN, fmt, *args)

    def error(self, fmt, *args):
        self.log(ERROR, fmt, *args)

    def trace(self, fmt, *args):
        self.log(TRACE, fmt, *args)

    def traceback(self, level=INFO):
        if (level > self.level):
            return
        lines = trim_traceback(traceback.format_exc())
        for line in lines:
            try:
                self.queue.put(((self.color | TRACEBACK), self.proc_id, line), False)
            except Exception:
                self.write_message(TRACEBACK, line)

    def close(self):
        self.queue.put(None)

    def write_message(self, level, text):
        if (level <= self.level):
            write_message_with_proc((self.color | level), self.proc_id, text)

def _message_loop(log_queue):
    common = set()
    while True:
        try:
            msg = log_queue.get()
            if (msg is None):
                return
            (level, proc_id, text) = msg
            if proc_id:
                write_message_with_proc(level, proc_id, text)
            elif ((level, text) not in common):
                write_message(level, text)
                common.add((level, text))
        except KeyboardInterrupt:
            pass

def trim_traceback(lines):
    trimmed = []
    for line in lines.splitlines()[(- 12):]:
        shortline = line.strip()
        try:
            if shortline.startswith('File'):
                shortline = ('"semmle' + shortline.split('semmle')[(- 1)])
            else:
                continue
        except Exception:
            pass
        trimmed.append(shortline)
    return trimmed
