
__ALL__ = ['load', 'ProjectLayout']
import collections
import re
from functools import total_ordering
import sys
if (sys.version >= '3'):
    unicode = str

def get_renamer(filename):
    layout = load(filename)

    def rename(path):
        renamed = layout.artificial_path(path)
        return (path if (renamed is None) else renamed)
    return rename

def load(filename):
    with open(filename, 'rb') as f:
        content = f.read().decode('utf-8')
        lines = [line.strip() for line in content.split('\n')]
    return ProjectLayout(lines)

def _escape_string_literal_for_regexp(literal, preserve):
    ESCAPE = '(){}[].^$+\\*?'

    def escape(char):
        if ((char in ESCAPE) and (not (char in preserve))):
            return ('\\' + char)
        else:
            return char
    return ''.join((escape(c) for c in literal))

class ProjectLayout(object):

    def __init__(self, lines):
        self._project = None
        self._rewrites = collections.OrderedDict()
        virtual = ''
        section = _Section()
        self._rewrites[virtual] = section
        num = 0
        for line in lines:
            num += 1
            if (not line):
                continue
            if (line[0] == '@'):
                if (self._project is not None):
                    raise _error('Multiple project names in project-layout', num)
                self._project = self._tail(line)
            elif (line[0] == '#'):
                virtual = self._tail(line)
                if (virtual in self._rewrites):
                    raise _error(('Duplicate virtual path prefix ' + virtual), num)
                section = _Section(virtual)
                self._rewrites[virtual] = section
            elif (line[0] == '-'):
                section.add(_Rewrite(self._tail(line), num))
            else:
                section.add(_Rewrite(line, num, virtual))

    @classmethod
    def _tail(cls, line):
        return line[1:].strip()

    def project_name(self, default=None):
        if (self._project is not None):
            return self._project
        if (default is not None):
            return default
        raise Exception('Project specificatino does not define a project name.')

    def sections(self):
        return self._rewrites.keys()

    def section_is_empty(self, section):
        if (section in self._rewrites):
            return self._rewrites[section].is_empty()
        raise Exception(('Section does not exist: ' + section))

    def rename_section(self, old, new):
        if (not (old in self._rewrites)):
            raise Exception(('Section does not exist: ' + old))
        section = self._rewrites.pop(old)
        section.rename(new)
        self._rewrites[new] = section

    def sub_layout(self, section_name):
        section = self._rewrites.get(section_name, None)
        if (section is None):
            raise Exception(('Section does not exist: ' + section))
        return section.to_layout()

    def artificial_path(self, path):
        prefixes = _Section.prefixes(path)
        for section in self._rewrites.values():
            rewrite = section.match(prefixes)
            rewritten = None
            if (rewrite is not None):
                rewritten = rewrite.rewrite(path)
            if (rewritten is not None):
                return rewritten
        return None

    def include_file(self, path):
        return (self.artificial_path(path) is not None)

class _Section(object):

    def __init__(self, virtual=''):
        self._virtual = virtual
        self._simple_rewrites = collections.OrderedDict()
        self._complex_rewrites = []

    def to_layout(self):
        result = []
        rewrites = []
        rewrites.extend(self._simple_rewrites.values())
        rewrites.extend(self._complex_rewrites)
        rewrites.sort()
        result.append(('@' + self._virtual))
        for rewrite in rewrites:
            result.append(unicode(rewrite))
        result.append('')
        return '\n'.join(result)

    def rename(self, new):
        self._virtual = new
        for rewrite in self._simple_rewrites.values():
            rewrite.virtual = new
        for rewrite in self._complex_rewrites:
            rewrite.virtual = new

    def add(self, rewrite):
        if rewrite.is_simple():
            self._simple_rewrites[rewrite.simple_prefix()] = rewrite
        else:
            self._complex_rewrites.append(rewrite)

    def is_empty(self):
        return ((not self._simple_rewrites) and (not self._complex_rewrites))

    @classmethod
    def prefixes(cls, path):
        result = [path]
        i = len(path)
        while (i > 1):
            i = path.rfind('/', 0, i)
            result.append(path[:i])
        result.append('/')
        return result

    def match(self, prefixes):
        best = None
        for prefix in prefixes:
            match = self._simple_rewrites.get(prefix, None)
            if (match is not None):
                if ((best is None) or (best._line < match._line)):
                    best = match
        for rewrite in reversed(self._complex_rewrites):
            if rewrite.matches(prefixes[0]):
                if ((best is None) or (best._line < rewrite._line)):
                    best = rewrite
                break
        return best

@total_ordering
class _Rewrite(object):
    _verify_stars = re.compile('.*(?:\\*\\*[^/].*|\\*\\*$|[^/]\\*\\*.*)')

    def __init__(self, path, line, virtual=None):
        if (virtual is None):
            exclude = path
            self._line = line
            self._original = ('-' + exclude)
            if (not exclude.startswith('/')):
                exclude = ('/' + exclude)
            if (exclude.find('//') != (- 1)):
                raise _error("Illegal '//' in exclude path", line)
            if self._verify_stars.match(exclude):
                raise _error("Illegal use of '**' in exclude path", line)
            if exclude.endswith('/'):
                exclude = exclude[0:(- 1)]
            self._pattern = self._compile_prefix(exclude)
            exclude = exclude.replace('//', '/')
            if ((len(exclude) > 1) and exclude.endswith('/')):
                exclude = exclude[0:(- 1)]
            self._simple = (None if (exclude.find('*') != (- 1)) else exclude)
        else:
            include = path
            self._line = line
            self._original = include
            if (not include.startswith('/')):
                include = ('/' + include)
            doubleslash = include.find('//')
            if (doubleslash != include.find('//')):
                raise _error("More than one '//' in include path (project-layout)", line)
            if self._verify_stars.match(include):
                raise _error("Illegal use of '**' in include path (project-layout)", line)
            if (not virtual.startswith('/')):
                virtual = ('/' + virtual)
            if virtual.endswith('/'):
                virtual = virtual[0:(- 1)]
            self._pattern = self._compile_prefix(include)
            include = include.replace('//', '/')
            if ((len(include) > 1) and include.endswith('/')):
                include = include[0:(- 1)]
            self._simple = (None if (include.find('*') != (- 1)) else include)
        self._virtual = virtual

    @classmethod
    def _compile_prefix(cls, pattern):
        pattern = _escape_string_literal_for_regexp(pattern, '*')
        if (pattern.find('//') != (- 1)):
            pattern = pattern.replace('//', '(/')
        else:
            pattern = ('(' + pattern)
        if pattern.endswith('/'):
            pattern = pattern[0:(- 1)]
        pattern = pattern.replace('/**', '-///-')
        pattern = pattern.replace('*', '[^/]*')
        pattern = pattern.replace('-///-', '(?:|/.*)')
        return re.compile((pattern + '(?:/.*|$))'))

    def is_simple(self):
        return (self._simple is not None)

    def simple_prefix(self):
        return self._simple

    def matches(self, path):
        return bool(self._pattern.match(path))

    def rewrite(self, path):
        if (self._virtual is None):
            return None
        matcher = self._pattern.match(path)
        if (not matcher):
            return None
        return (self._virtual + matcher.group(1))

    def __unicode__(self):
        return self._original

    def __lt__(self, other):
        return (self._line < other._line)
    __hash__ = None

def _error(message, line):
    raise Exception(('%s (line %d)' % (message, line)))
