
from semmle.python import ast
from semmle.python.passes.ast_pass import iter_fields, ASTVisitor
__all__ = ['Labeller']

class SymbolTable(ASTVisitor):

    def __init__(self, scope):
        self.definitions = set()
        self.uses = set()
        self.declared_as_global = set()
        self.declared_as_nonlocal = set()
        for (_, _, child) in iter_fields(scope):
            self.visit(child)

    def visit_Class(self, node):
        pass

    def visit_Function(self, node):
        pass

    def visit_Name(self, node):
        name = node.variable.id
        if isinstance(node.ctx, ast.Load):
            self.uses.add(name)
        elif isinstance(node.ctx, (ast.Store, ast.Param, ast.Del)):
            self.definitions.add(name)
        else:
            raise Exception(('Unknown context for name: %s' % node.ctx))

    def visit_Global(self, node):
        self.declared_as_global.update(node.names)

    def visit_NonLocal(self, node):
        self.declared_as_nonlocal.update(node.names)

    def is_bound(self, name):
        declared_free = ((name in self.declared_as_global) or (name in self.declared_as_nonlocal))
        return ((name in self.definitions) and (not declared_free))

class _LabellingContext(ASTVisitor):

    def __init__(self, scope, module=None, outer=None):
        self.symbols = SymbolTable(scope)
        self.scope = scope
        self.outer = outer
        if (module is None):
            module = scope
        self.module = module

    def label(self):
        self.visit(self.module)

    def visit_Function(self, node):
        sub_context = _LabellingContext(node, self.module, self)
        for (_, _, child) in iter_fields(node):
            sub_context.visit(child)
    visit_Class = visit_Function

    def visit_Variable(self, node):
        if (node.scope is not None):
            return
        name = node.id
        if (name in self.symbols.declared_as_global):
            node.scope = self.module
        elif self.symbols.is_bound(name):
            node.scope = self.scope
        else:
            outer = self.outer
            while (outer is not None):
                if isinstance(outer.scope, ast.Class):
                    pass
                elif outer.symbols.is_bound(name):
                    node.scope = outer.scope
                    break
                outer = outer.outer
            else:
                node.scope = self.module

class Labeller(object):

    def apply(self, module):
        if (module.ast is None):
            return
        _LabellingContext(module.ast).label()
